/* SIMPLE - Simple Is a Macro Processing Language Element */
/* Copyright (c) 1998 David A. Madore (david.madore@ens.fr) */

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
 * the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * Handle user-defined macros (defining them, calling them, etc.).
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "simple.h"
#include "token.h"

typedef struct def_entry {
  command who; /* What command is being defined */
  token_list what; /* Definition of that token */
} def_entry;

static def_entry *user_table = NULL;
static int user_table_len = 0;

void
replace_user(int entry,token_list *definition)
     /* Modify the definition of a user token */
{
  free_list(&user_table[entry].what);
  copy_list(&user_table[entry].what,definition);
}

void
create_user(token_list *definition,command cmd)
     /* Create a new user token */
{
  if ((user_table=REALLOC(user_table,(++user_table_len)*sizeof(def_entry)))
      ==NULL)
    fatal(0,"Out of memory!");
  user_table[user_table_len-1].who=cmd;
  make_empty_list(&user_table[user_table_len-1].what);
  replace_user(user_table_len-1,definition);
}

void
define_user(token_list *definition,command cmd)
     /* Define a user token, checking to see whether it already exists */
{
  int i;
  for (i=0;i<user_table_len;i++) {
    if (cmd==user_table[i].who) {
      replace_user(i,definition);
      return;
    }
  }
  create_user(definition,cmd);
}

void
expand_user(token_list *expansion,token_list *definition,
	   int argc,token_list *argv)
     /* Expand user command definition, putting the result in
      * expansion, the number of arguments being argc and their values
      * given by argv. */
{
  int i,j;
  token_list tmplist;
  char *tmpstring,*tmpstring2;
  int cnt,cnt2;
  for (i=0;i<definition->size;i++) {
    if (definition->l[i]!=AT_SIGN)
      append_token(expansion,definition->l[i]);
    else {
      make_empty_list(&tmplist);
      for (i++;(i<definition->size)&&(definition->l[i]!=AT_SIGN);i++)
	append_token(&tmplist,definition->l[i]);
      if (i==definition->size)
	fatal(0,"Bad parameter construct (missing @).");
      tmpstring=token_list_to_string(&tmplist);
      free_list(&tmplist);
      if (strcmp(tmpstring,"")==0) {
	append_token(expansion,AT_SIGN); /* Whatever that's good for... */
      } else if (strcmp(tmpstring,"?")==0) {
	tmpstring2=malloc(30);
#if defined(HAVE_SNPRINTF)
	if (snprintf(tmpstring2,30,"%d",argc-1)==30)
	  fatal(1,"Ooops! not enough characters...");
#else
	sprintf(tmpstring2,"%d",argc-1);
#endif
	for (j=0;tmpstring2[j];j++)
	  append_token(expansion,tmpstring2[j]);
	FREE(tmpstring2);
      } else if (tmpstring[0]=='.') {
	if (strcmp(tmpstring,".")==0)
	  cnt=1;
	else if (sscanf(tmpstring+1,"%i",&cnt)<1)
	  fatal(0,"Bad parameter construct (didn't understand %s).",tmpstring);
	if (cnt<0) cnt+=argc;
	for (cnt2=cnt;(cnt2>=0)&&(cnt2<argc);cnt2++) {
	  append_token(expansion,NEXT_PARAM);
	  append_list(expansion,argv+cnt2);
	}
      } else if (tmpstring[0]==';') {
	if (strcmp(tmpstring,";")==0)
	  cnt=1;
	else if (sscanf(tmpstring+1,"%i",&cnt)<1)
	  fatal(0,"Bad parameter construct (didn't understand %s).",tmpstring);
	if (cnt<0) cnt+=argc;
	for (cnt2=cnt;(cnt2>=0)&&(cnt2<argc);cnt2++) {
	  append_token(expansion,NEXT_PARAM);
	  for (j=0;j<argv[cnt2].size;j++) {
	    append_token(expansion,QUOTE_NEXT);
	    append_token(expansion,argv[cnt2].l[j]);
	  }
	}
      } else if (tmpstring[0]==',') {
	int k,l;
	for (k=1;tmpstring[k]==',';k++);
	if (sscanf(tmpstring+k,"%i",&cnt)<1)
	  fatal(0,"Bad parameter construct (didn't understand %s).",tmpstring);
	if (cnt<0) cnt+=argc;
	if ((cnt>=0)&&(cnt<argc)) {
	  for (j=0;j<argv[cnt].size;j++) {
	    for (l=0;l<((1<<k)-1);l++)
	      append_token(expansion,QUOTE_NEXT);
	    append_token(expansion,argv[cnt].l[j]);
	  }
	}
      } else {
	if (sscanf(tmpstring,"%i",&cnt)<1)
	  fatal(0,"Bad parameter construct (didn't understand %s).",tmpstring);
	if (cnt<0) cnt+=argc;
	if ((cnt>=0)&&(cnt<argc))
	  append_list(expansion,argv+cnt);
	else /* Append the empty list: no-op. */;
      }
      FREE(tmpstring);
    }
  }
}

void
call_user(token_list *expansion,command cmd,int argc,token_list *argv)
     /* Expand user command cmd looking up its definition, and put the
      * result in expansion, the number of arguments being argc and
      * their values given by argv. */
{
  int i;
  for (i=0;i<user_table_len;i++) {
    if (cmd==user_table[i].who) {
      expand_user(expansion,&user_table[i].what,argc,argv);
      return;
    }
  }
  fatal(0,"Undefined command.");
}

void
def_of_user(token_list *expansion,command cmd)
     /* Simply copy the definition of user command cmd.  This is used
      * to implement <defof>. */
{
  int i;
  for (i=0;i<user_table_len;i++) {
    if (cmd==user_table[i].who) {
      copy_list(expansion,&user_table[i].what);
      return;
    }
  }
  fatal(0,"Undefined command.");
}
